/* MKS Plugin.cpp */

#include "rb_plugin.h"
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "typedef.h"
#include "stringStuff.h"
#include "fractions.h"
#include "MKS.h"

#define kCFStringEncodingUTF8 0x08000100

// WinHeader.h

//static long myPrecision;

//extern REALclassDefinition		FractionClass;

extern REALclassDefinition			MKSClass;

//extern REALclassDefinition		BadArgumentExceptionClass;
//extern REALclassDefinition		DivideByZeroExceptionClass;
//extern REALclassDefinition		IntegerOverflowExceptionClass;
//extern REALclassDefinition		UnitMismatchClass;

#define BadArgumentName "MKSBadArgumentException"
#define DivideByZeroName "MKSDivideByZeroException"
#define UnitMismatchName "MKSUnitMismatchException"

/* -------------------------------------
 MKSConstructor
 
 Called by REALbasic when a new MKS object is instantiated.
 
 instance = the new MKS object
 ------------------------------------- */

static void MKSConstructor(REALobject instance)
{
	MKSdata		*data;
	long		i;
	
	// Get the class data
	data = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (data == nil)
		return;
	
	data->value = 0;
	
	for(i=0;i<numUnits;i++)
	{
		data->unit[i].num = 0;
		data->unit[i].den = 1;
	}
	
}/* MKSConstructor */


static void NewMKSD(REALobject instance, double value)
{
	MKSdata		*data;
	long		i;
	
	// Get the class data
	data = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (data == nil)
		return;
	
	data->value = value;
	
	for(i=0;i<numUnits;i++)
	{
		data->unit[i].num = 0;
		data->unit[i].den = 1;
	}
	
}/* NewMKSD */


static void NewMKSS(REALobject instance, REALstring CStr)
{
	MKSdata			*data;
	long			i, length;
	char			*dataString;
	
	// Get the class data
	data = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (data == nil)
		return;
	
	// default
	data->value = 0;
	
	for(i=0;i<numUnits;i++)
	{
		data->unit[i].num = 0;
		data->unit[i].den = 1;
	}
	
	// now get the fraction from CStr
	length = CStr->Length();
	
	// if length==0 make it a zero MKS and return
	if(length==0)
		return;
	
	// length!=0
	dataString = (char *)malloc((length+1)*sizeof(char));
	memcpy(dataString, (char*)CStr->CString(), length); // to, from
	dataString[length] = 0; // makes it a C string
	
	//StrToMKS(*data, dataString);
	
	if(!StrToMKS(*data, dataString))
	{
		REALobject err = REALnewInstance("MKSBadArgumentException");
		REALRaiseException(err);
	}
	
	free(dataString);
	
}/* NewMKSS */


/* -------------------------------------
 MKSDestructor
 
 Called by REALbasic when a MKS object is destroyed.
 
 instance = the MKS object being destroyed
 ------------------------------------- */

static void MKSDestructor(REALobject instance)
{
	
}/* MKSDestructor */


static REALobject MKSNegate(REALobject instance)
{
	REALobject		z;
	MKSdata			*data, *zData;
	long			i;
	
	if(instance==nil)
		return instance;
	
	data = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (data == nil)
		return instance;
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = -data->value;
	
	for(i=0;i<numUnits;i++)
	{
		zData->unit[i].num = data->unit[i].num;
		zData->unit[i].den = data->unit[i].den;
	}
	
	return z;
	
}/* MKSNegate */


static double MKSvalue(REALobject instance)
{
	MKSdata		*data;
	
	if(instance==nil)
	{
		return 0;
	}
	
	data = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (data == nil)
		return 0;
	
	return data->value;
	
}/* MKSvalue */


static REALstring MKSstr(REALobject instance)
{
	MKSdata			*data;
	long			length;
	char			myString[1024];
	
	if(instance==nil)
	{
		REALstring str=REALBuildString (nil, 0);
		return str;
	}
	
	data = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (data == nil)
		strcpy(myString, "");
	else
		MKStoStr(myString, *data);
	
	length = strlen(myString);
	REALstring str=REALBuildString (nil, length);
	memcpy ((char*)str->CString(), myString, length); // to, from
	
	REALSetStringEncoding(str, kCFStringEncodingUTF8); // kCFStringEncodingUTF8=0x08000100
	
	return str;
	
}/* MKSstr */


static REALstring MKSstr2(REALobject instance, REALstring userUnitString)
{
	MKSdata			*data;
	long			length;
	char			myString[1024], unitString[256];
	
	REALstring str=REALBuildString (nil, 0);
	
	if(instance==nil)
		return str;
	
	data = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	length = userUnitString->Length();
	if(length==0)
		return str;
	
	memcpy(unitString, (char*)userUnitString->CString(), length);
	unitString[length] = 0;
	
	if (data == nil)
		strcpy(myString, "");
	else
	{
		if(!MKStoStr2(myString, *data, unitString))
		{
			REALobject err = REALnewInstance("MKSBadArgumentException");
			REALRaiseException(err);
		}
	}
	
	length = strlen(myString);
	REALstring str2=REALBuildString (nil, length);
	memcpy ((char*)str2->CString(), myString, length); // to, from
	
	REALSetStringEncoding(str2, kCFStringEncodingUTF8); // kCFStringEncodingUTF8=0x08000100
	
	return str2;
	
}/* MKSstr2 */


static REALobject MKSAddMKS(REALobject instance, REALobject y)
{
	MKSdata			*xData, *yData, *zData;
	REALobject		z;
	
	if(instance==nil || y==nil)
		return instance;
	
	xData = (MKSdata *) REALGetClassData(instance, &MKSClass);
	yData = (MKSdata *) REALGetClassData(y, &MKSClass);
	
	if (xData == nil || yData == nil)
		return instance;
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	if(!addMKS(*zData, *xData, *yData))
	{
		REALobject err = REALnewInstance("MKSUnitMismatchException");
		REALRaiseException(err);
		return instance;
	}
	
	return z;
	
}/* MKSAddMKS */

// can add a double only if MKS object is dimensionless
static REALobject MKSAddD(REALobject instance, double y)
{
	MKSdata			*xData, *zData;
	REALobject		z;
	long			i;
	
	if(instance==nil)
		return instance;
	
	xData = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (xData == nil)
		return instance;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != 0)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return instance;
		}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = xData->value + y;
	
	for(i=0;i<numUnits;i++)
	{
		zData->unit[i].num = xData->unit[i].num;
		zData->unit[i].den = xData->unit[i].den;
	}
	
	return z;
	
}/* MKSAddD */


// can subtract a double only if MKS object is dimensionless
static REALobject MKSSubMKS(REALobject instance, REALobject y)
{
	MKSdata			*xData, *yData, *zData;
	REALobject		z;
	
	if(instance==nil || y==nil)
		return instance;
	
	xData = (MKSdata *) REALGetClassData(instance, &MKSClass);
	yData = (MKSdata *) REALGetClassData(y, &MKSClass);
	
	if (xData == nil || yData == nil)
		return instance;
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	if(!subMKS(*zData, *xData, *yData))
	{
		REALobject err = REALnewInstance("MKSUnitMismatchException");
		REALRaiseException(err);
		return instance;
	}
	
	return z;
	
}/* MKSSubMKS */


// can subtract a double only if MKS object is dimensionless
static REALobject MKSSubD(REALobject instance, double y)
{
	MKSdata			*xData, *zData;
	REALobject		z;
	long			i;
	
	if(instance==nil)
		return instance;
	
	xData = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (xData == nil)
		return instance;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != 0)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return instance;
		}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = xData->value - y;
	
	for(i=0;i<numUnits;i++)
	{
		zData->unit[i].num = xData->unit[i].num;
		zData->unit[i].den = xData->unit[i].den;
	}
	
	return z;
	
}/* MKSSubD */


static REALobject DSubMKS(REALobject instance, long y)
{
	MKSdata			*xData, *zData;
	REALobject		z;
	long			i;
	
	if(instance==nil)
		return instance;
	
	xData = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (xData == nil)
		return instance;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != 0)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return instance;
		}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = y - xData->value;
	
	for(i=0;i<numUnits;i++)
	{
		zData->unit[i].num = xData->unit[i].num;
		zData->unit[i].den = xData->unit[i].den;
	}
	
	return z;
	
}/* DSubMKS */


static REALobject MKSMulMKS(REALobject instance, REALobject y)
{
	MKSdata			*xData, *yData, *zData;
	REALobject		z;
	
	if(instance==nil || y==nil)
		return instance;
	
	xData = (MKSdata *) REALGetClassData(instance, &MKSClass);
	yData = (MKSdata *) REALGetClassData(y, &MKSClass);
	
	if (xData == nil || yData == nil)
		return instance;
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	mulMKS(*zData, *xData, *yData);
	
	return z;
	
}/* MKSMulMKS */


static REALobject MKSMulD(REALobject instance, double y)
{
	MKSdata			*xData, *zData;
	REALobject		z;
	long			i;
	
	if(instance==nil)
		return instance;
	
	xData = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (xData == nil)
		return instance;
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = y*xData->value;
	
	for(i=0;i<numUnits;i++)
	{
		zData->unit[i].num = xData->unit[i].num;
		zData->unit[i].den = xData->unit[i].den;
	}
	
	return z;
	
}/* MKSMulD */


static REALobject MKSDivMKS(REALobject instance, REALobject y)
{
	MKSdata			*xData, *yData, *zData;
	REALobject		z;
	
	if(instance==nil || y==nil)
		return instance;
	
	xData = (MKSdata *) REALGetClassData(instance, &MKSClass);
	yData = (MKSdata *) REALGetClassData(y, &MKSClass);
	
	if (xData == nil || yData == nil)
		return instance;
	
	if(yData->value==0)
	{
		REALobject err = REALnewInstance("MKSDivideByZeroException");
		REALRaiseException(err);
		return instance;
	}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	divMKS(*zData, *xData, *yData);
	
	return z;
	
}/* MKSDivMKS */


static REALobject MKSDivD(REALobject instance, double y)
{
	MKSdata			*xData, *zData;
	REALobject		z;
	long			i;
	
	if(instance==nil)
		return instance;
	
	if(y==0)
	{
		REALobject err = REALnewInstance("MKSDivideByZeroException");
		REALRaiseException(err);
		return instance;
	}
	
	xData = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (xData == nil)
		return instance;
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = xData->value/y;
	
	for(i=0;i<numUnits;i++)
	{
		zData->unit[i].num = xData->unit[i].num;
		zData->unit[i].den = xData->unit[i].den;
	}
	
	return z;
	
}/* MKSDivD */


static REALobject DDivMKS(REALobject instance, double x)
{
	MKSdata			*yData, *zData;
	REALobject		z;
	long			i;
	
	if(instance==nil)
		return instance;
	
	yData = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if (yData == nil)
		return instance;
	
	if(yData->value==0)
	{
		REALobject err = REALnewInstance("MKSDivideByZeroException");
		REALRaiseException(err);
		return instance;
	}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = x/yData->value;
	
	for(i=0;i<numUnits;i++)
	{
		zData->unit[i].num = -yData->unit[i].num;
		zData->unit[i].den = yData->unit[i].den;
	}
	
	return z;
	
}/* DDivMKS */


static REALobject myPowMKSL(REALobject instance, long y)
{
	REALobject	z;
	MKSdata		*data, *zData;
	
	if(instance==nil)
		return instance;
	
	data = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if(data==nil)
		return instance;
	
	if(data->value==0 && y==0)
	{
		REALobject err = REALnewInstance("MKSBadArgumentException");
		REALRaiseException(err);
		return instance;
	}
	
	if(data->value==0 && y<0)
	{
		REALobject err = REALnewInstance("MKSDivideByZeroException");
		REALRaiseException(err);
		return instance;
	}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	if(zData==nil)
		return instance;
	
	myPowMKS(*zData, *data, y);
	
	return z;
	
}/* myPowMKSL */


static REALobject myPowMKSF(REALobject instance, REALstring y)
{
	REALobject	z;
	MKSdata		*data, *zData;
	char		unitString[256];
	long		length;
	fract		yFract;
	
	if(instance==nil)
		return instance;
	
	data = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if(data==nil)
		return instance;
	
	length = y->Length();
	if(length==0)
		return instance;
	
	memcpy(unitString, (char*)y->CString(), length);
	unitString[length] = 0;
	
	if(!StrToFract(yFract, unitString))
	{
		REALobject err = REALnewInstance("MKSBadArgumentException");
		REALRaiseException(err);
		return instance;
	}
	
	if(data->value==0 && yFract.num==0)
	{
		REALobject err = REALnewInstance("MKSBadArgumentException");
		REALRaiseException(err);
		return instance;
	}
	
	if(data->value==0 && yFract.num<0)
	{
		REALobject err = REALnewInstance("MKSDivideByZeroException");
		REALRaiseException(err);
		return instance;
	}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	if(zData==nil)
		return instance;
	
	myPowMKS2(*zData, *data, yFract);
	
	return z;
	
}/* myPowMKSF */


static int CompareMKS(REALobject instance, REALobject x)
{
	MKSdata		*data, *xData;
	long		i;
	
	if(instance==nil || x==nil)
		return 0;
	
	data = (MKSdata *) REALGetClassData(instance, &MKSClass);
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if(data==nil || xData==nil)
		return 0;
	
	for(i=0;i<numUnits;i++)
		if(!isEqualFract(data->unit[i], xData->unit[i]))
			return 1;
	
	if(data->value<xData->value)
		return -1;
	
	if(data->value>xData->value)
		return 1;
	
	return 0;
	
}/* CompareMKS */


static int CompareMKSD(REALobject instance, double x)
{
	MKSdata		*data;
	long		i;
	
	if(instance==nil)
		return 0;
	
	data = (MKSdata *) REALGetClassData(instance, &MKSClass);
	
	if(data==nil)
		return 0;
	
	for(i=0;i<numUnits;i++)
		if(data->unit[i].num != 0)
			return 1;
	
	if(data->value<x)
		return -1;
	
	if(data->value>x)
		return 1;
	
	return 0;
	
}/* CompareMKSD */


// REALbasic class & method definitions for the MKS object

static REALmethodDefinition MKSMethods[] = {
	// defines methods:
	{ (REALproc) MKSvalue, REALnoImplementation, "Val() As Double", REALconsoleSafe },
	{ (REALproc) MKSstr, REALnoImplementation, "Str() as String", REALconsoleSafe },
	{ (REALproc) MKSstr2, REALnoImplementation, "Str(userUnitString As String) as String", REALconsoleSafe },
	{ (REALproc) MKSConstructor, REALnoImplementation, "Constructor()", REALconsoleSafe },
	{ (REALproc) NewMKSD, REALnoImplementation, "Constructor(Value As Double)", REALconsoleSafe },
	{ (REALproc) NewMKSS, REALnoImplementation, "Constructor(inputString As String)", REALconsoleSafe },
	{ (REALproc) MKSNegate, REALnoImplementation, "Operator_Negate() As MKS", REALconsoleSafe },
	{ (REALproc) MKSAddMKS, REALnoImplementation, "Operator_Add(y As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) MKSAddD, REALnoImplementation, "Operator_Add(y As Double) As MKS", REALconsoleSafe },
	{ (REALproc) MKSAddD, REALnoImplementation, "Operator_AddRight(x As Double) As MKS", REALconsoleSafe },
	{ (REALproc) MKSSubMKS, REALnoImplementation, "Operator_Subtract(y As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) MKSSubD, REALnoImplementation, "Operator_Subtract(y As Double) As MKS", REALconsoleSafe },
	{ (REALproc) DSubMKS, REALnoImplementation, "Operator_SubtractRight(x As Double) As MKS", REALconsoleSafe },
	{ (REALproc) MKSMulMKS, REALnoImplementation, "Operator_Multiply(y As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) MKSMulD, REALnoImplementation, "Operator_Multiply(y As Double) As MKS", REALconsoleSafe },
	{ (REALproc) MKSMulD, REALnoImplementation, "Operator_MultiplyRight(x As Double) As MKS", REALconsoleSafe },
	{ (REALproc) MKSDivMKS, REALnoImplementation, "Operator_Divide(y As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) MKSDivD, REALnoImplementation, "Operator_Divide(y As Double) As MKS", REALconsoleSafe },
	{ (REALproc) DDivMKS, REALnoImplementation, "Operator_DivideRight(x As Double) As MKS", REALconsoleSafe },
	{ (REALproc) myPowMKSL, REALnoImplementation, "Operator_Power(x As Integer) As MKS", REALconsoleSafe },
	{ (REALproc) myPowMKSF, REALnoImplementation, "Operator_Power(x As String) As MKS", REALconsoleSafe },
	{ (REALproc) CompareMKS, REALnoImplementation, "Operator_Compare(x As MKS) As Integer", REALconsoleSafe },
	{ (REALproc) CompareMKSD, REALnoImplementation, "Operator_Compare(x As Double) As Integer", REALconsoleSafe },
};


REALclassDefinition MKSClass = {
	kCurrentREALControlVersion,
	"MKS",                    			// name of class
	nil,                                    // no superclasses
	sizeof(MKSdata),                      // size of our data
	0,																			// --reserved--
	(REALproc) MKSConstructor,     				// constructor
	(REALproc) MKSDestructor,							// destructor
	nil,            												// properties
	0,																			// property count
	MKSMethods,               					// methods
	sizeof(MKSMethods) / sizeof(REALmethodDefinition),	// method count
	nil,													// Events
	0,
	nil,												// bindings
	0,
	nil,												// constants
	0,
	REALconsoleSafe,									// flags
	nil,												// shared properties
	0,
	nil,												// shared methods
	0,
};

// the following pragma is essential here

#pragma mark -


// other methods


static double ConvToDouble(REALobject x)
{
	MKSdata			*xData;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return 0;
	
	return xData->value;
	
}/* ConvToDouble */



static REALobject AbsMKS(REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *zData;
	long			i;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	if(xData->value>=0)
		zData->value = xData->value;
	else
		zData->value = -xData->value;
	
	for(i=0;i<numUnits;i++)
	{
		zData->unit[i].num = xData->unit[i].num;
		zData->unit[i].den = xData->unit[i].den;
	}
	
	return z;
	
}/* AbsMKS */



static REALobject SqrMKS(REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *zData;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	sqrMKS(*zData, *xData);
	
	return z;
	
}/* SqrMKS */


static REALobject SqrtMKS(REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *zData;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	if(xData->value<0)
	{
		REALobject err = REALnewInstance("MKSBadArgumentException");
		REALRaiseException(err);
		return x;
	}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	sqrtMKS(*zData, *xData);
	
	return z;
	
}/* SqrtMKS */


static REALobject ExpMKS(REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *zData;
	long			i;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != 0)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return x;
		}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = exp(xData->value);
	
	return z;
	
}/* ExpMKS */


static REALobject LogMKS(REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *zData;
	long			i;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != 0)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return x;
		}
	
	if(xData->value<=0)
	{
		REALobject err = REALnewInstance("MKSBadArgumentException");
		REALRaiseException(err);
		return x;
	}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = log(xData->value);
	
	return z;
	
}/* LogMKS */


static REALobject SinMKS(REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *zData;
	long			i;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != 0)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return x;
		}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = sin(xData->value);
	
	return z;
	
}/* SinMKS */


static REALobject CosMKS(REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *zData;
	long			i;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != 0)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return x;
		}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = cos(xData->value);
	
	return z;
	
}/* CosMKS */


static REALobject TanMKS(REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *zData;
	long			i;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != 0)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return x;
		}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = tan(xData->value);
	
	return z;
	
}/* TanMKS */


static REALobject ASinMKS(REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *zData;
	long			i;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != 0)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return x;
		}
	
	if(abs(xData->value)>1)
	{
		REALobject err = REALnewInstance("MKSBadArgumentException");
		REALRaiseException(err);
		return x;
	}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = asin(xData->value);
	
	return z;
	
}/* ASinMKS */


static REALobject ACosMKS(REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *zData;
	long			i;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != 0)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return x;
		}
	
	if(abs(xData->value)>1)
	{
		REALobject err = REALnewInstance("MKSBadArgumentException");
		REALRaiseException(err);
		return x;
	}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = acos(xData->value);
	
	return z;
	
}/* ACosMKS */


static REALobject ATanMKS(REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *zData;
	long			i;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != 0)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return x;
		}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = atan(xData->value);
	
	return z;
	
}/* ATanMKS */


static REALobject ATan2MKS(REALobject y, REALobject x)
{
	REALobject		z;
	MKSdata			*xData, *yData, *zData;
	long			i;
	
	if(x==nil)
		return x;
	
	xData = (MKSdata *) REALGetClassData(x, &MKSClass);
	
	if (xData == nil)
		return x;
	
	if(y==nil)
		return y;
	
	yData = (MKSdata *) REALGetClassData(y, &MKSClass);
	
	if (yData == nil)
		return y;
	
	for(i=0;i<numUnits;i++)
		if(xData->unit[i].num != yData->unit[i].num)
		{
			REALobject err = REALnewInstance("MKSUnitMismatchException");
			REALRaiseException(err);
			return x;
		}
	
	z = REALnewInstance("MKS");
	zData = (MKSdata *) REALGetClassData(z, &MKSClass);
	
	zData->value = atan2(yData->value, xData->value);
	
	return z;
	
}/* ATan2MKS */


static void MKSsetPrec(int prec)
{
	setPrec(prec);
	
}/* MKSsetPrec */


static int MKSgetPrec()
{
	return getPrec();
	
}/* MKSgetPrec */


static REALmethodDefinition otherMKSMethods[] = {
	{ (REALproc) MKSvalue, REALnoImplementation, "Val(x As MKS) As Double", REALconsoleSafe },
	{ (REALproc) MKSstr, REALnoImplementation, "Str(x As MKS) as String", REALconsoleSafe },
	{ (REALproc) MKSstr2, REALnoImplementation, "Str(x As MKS, userUnitString As String) as String", REALconsoleSafe },
	{ (REALproc) ConvToDouble, REALnoImplementation, "ConvToDouble(x As MKS) As Double", REALconsoleSafe },
	{ (REALproc) AbsMKS, REALnoImplementation, "Abs(x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) SqrMKS, REALnoImplementation, "Sqr(x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) SqrtMKS, REALnoImplementation, "Sqrt(x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) ExpMKS, REALnoImplementation, "Exp(x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) LogMKS, REALnoImplementation, "Log(x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) SinMKS, REALnoImplementation, "Sin(x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) CosMKS, REALnoImplementation, "Cos(x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) TanMKS, REALnoImplementation, "Tan(x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) ASinMKS, REALnoImplementation, "Asin(x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) ACosMKS, REALnoImplementation, "Acos(x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) ATanMKS, REALnoImplementation, "Atan(x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) ATan2MKS, REALnoImplementation, "Atan2(y As MKS, x As MKS) As MKS", REALconsoleSafe },
	{ (REALproc) myPowMKSL, REALnoImplementation, "Pow(x As MKS, y As Integer) As MKS", REALconsoleSafe },
	{ (REALproc) myPowMKSF, REALnoImplementation, "Pow(x As MKS, y As String) As MKS", REALconsoleSafe },
	{ (REALproc) MKSsetPrec, REALnoImplementation, "MKSsetPrec(prec As Integer)", REALconsoleSafe },
	{ (REALproc) MKSgetPrec, REALnoImplementation, "MKSgetPrec() As Integer", REALconsoleSafe },
};


REALclassDefinition BadArgumentExceptionClass = {
	kCurrentREALControlVersion,
	BadArgumentName,                    			// name of class
	"RuntimeException",
	// other fields left nil -- no events etc.
};

REALclassDefinition DivideByZeroExceptionClass = {
	kCurrentREALControlVersion,
	DivideByZeroName,                    			// name of class
	"RuntimeException",
	// other fields left nil -- no events etc.
};

REALclassDefinition UnitMismatchClass = {
	kCurrentREALControlVersion,
	UnitMismatchName,                			// name of class
	"RuntimeException",
	// other fields left nil -- no events etc.
};

void PluginEntry(void)
{
	int		i;
	
	SetClassConsoleSafe(&MKSClass);
	SetClassConsoleSafe(&BadArgumentExceptionClass);
	SetClassConsoleSafe(&DivideByZeroExceptionClass);
	SetClassConsoleSafe(&UnitMismatchClass);
	
	REALRegisterClass(&MKSClass);
	REALRegisterClass(&BadArgumentExceptionClass);
	REALRegisterClass(&DivideByZeroExceptionClass);
	REALRegisterClass(&UnitMismatchClass);
	
	for (i = 0; i < sizeof(otherMKSMethods) / sizeof(REALmethodDefinition); ++i)
		REALRegisterMethod(&otherMKSMethods[i]);
}
